# -*- coding: utf-8 -*-

from odoo import api, fields, models, _
from odoo.addons.industry_fsm.models.project_task import Task
from odoo.exceptions import ValidationError

import html2text

import logging
_logger = logging.getLogger(__name__)

native_action_fsm_validate = Task.action_fsm_validate


class TaskType(models.Model):
    _inherit = "project.task.type"
    
    allowed_group_ids = fields.Many2many('res.groups', string="Allowed Group Users")

class Task(models.Model):
    _inherit = "project.task"
    
    ap_date_inter_confirme_par_le_client = fields.Boolean(string="Date confirmée par le client")
    ap_n_intervention = fields.Char(string="N° d'intervention")
    ap_n_bon_de_commande = fields.Many2one('sale.order', string="N° Bon de commande")
    tech_timer_id = fields.One2many('timer.timer', compute='_compute_tech_timer_id', search='_search_tech_timer_id')
    tech_timer_start = fields.Datetime(related='tech_timer_id.timer_start')
    timer_start_ap = fields.Datetime("Début intervention réelle")
    timer_stop_ap = fields.Datetime("Fin intervention réelle")
    display_termine_nf = fields.Boolean(compute='_compute_display_terminer_buttons')
    
    
    def _search_tech_timer_id(self, operator, value):
        timers = self.env['timer.timer'].search([
            ('id', operator, value),            
        ])
        return [('id', 'in', timers.mapped('res_id'))]

    def _compute_tech_timer_id(self):
        """ Get the timers according these conditions
            :res_id is the current record
            :res_model is the current model
            limit=1 by security but the search should never have more than one record
        """
        for record in self:
            record.tech_timer_id = self.env['timer.timer'].search([
                ('res_id', '=', record.id),
                ('res_model', '=', record._name)
            ], limit=1)
    
    @api.model
    def create(self, vals):
        if 'helpdesk_ticket_id' in vals:
            ticket = self.env['helpdesk.ticket'].browse(vals['helpdesk_ticket_id'])
            if ticket:
                vals['description'] = ticket.description

        ### on incrémente un numéro d'intervention pour les services sur site ###
        if vals['project_id'] == self.env.ref('access_protection.project_tech_installation').id \
            or vals['project_id'] == self.env.ref('access_protection.project_service_technique').id:
                vals['ap_n_intervention'] = self.env['ir.sequence'].next_by_code('ref_intervention')
            
        res = super(Task, self).create(vals) 
        
        if vals['project_id'] == self.env.ref('access_protection.project_service_technique').id:
            ### on envoi un mail à l'équipe technique
            _logger.info("envoi mail")
        
        return res
    
    def write(self, vals):                        
        if 'stage_id' in vals:        
            if (vals.get('stage_id') == self.env.ref('access_protection.sav_termine_nf', False).id or vals.get('stage_id') == self.env.ref('access_protection.sav_termine_f', False).id) \
            and not self.env.user.has_group('access_protection.group_user_ap_administratif'):
                raise ValidationError(_("Vous n'avez pas le droit de sélectionner ce statut."))
        
        ## Terminée ###
        if vals.get('worksheet_signature'):
            for task in self:
                if task.stage_id == self.env.ref('access_protection.sav_planifie', False):
                    vals['color'] = 10
                    if task.project_id == self.env.ref('access_protection.project_service_technique'):
                        vals['stage_id'] = self.env.ref('access_protection.sav_effectue', False).id                    
        
        res = super(Task, self).write(vals)
        
        return res
        
    def _fsm_ensure_sale_order(self):
        sale_order = super(Task, self)._fsm_ensure_sale_order()
         
        """ Projet Assistance - Service technique on ajoute le déplacement et on modifie la position fiscale"""
        if self.project_id == self.env.ref('access_protection.project_service_technique'):
            self.sale_order_id.fiscal_position_id = self.env.ref('access_protection.fiscal_position_sav').id

            product_deplacement = self.env.ref('access_protection.product_template_deplacement')
            if not self.env['sale.order.line'].search([('order_id', '=', self.sale_order_id.id), ('product_id', '=', product_deplacement.id)]):
                if product_deplacement:
                    self.env['sale.order.line'].sudo().create({
                        'order_id': self.sale_order_id.id,
                        'product_id': product_deplacement.id,
                        'project_id': self.project_id.id,
                        'task_id': self.id,
                        'product_uom_qty': 1,
                        'product_uom': product_deplacement.uom_id.id,
                    })

        line_service = self.env['sale.order.line'].search([('order_id', '=', self.sale_order_id.id), ('product_id', '=', self.project_id.timesheet_product_id.id)])
        if line_service:
            if self.description:
                line_service.name = html2text.html2text(self.description)
            for ts in self.timesheet_ids:
                line_service.name += "\n" + ts.name
        
        return sale_order
        
    def action_fsm_validate(self):
        """ If allow billable on task, timesheet product set on project and user has privileges :
            Create SO confirmed with time and material.
        """
        native_action_fsm_validate(self)
        for task in self.filtered(lambda task: (task.allow_timesheets or task.allow_material)):
            if not task.sale_line_id and not task.timesheet_ids: # Prevent creating a SO if there are no products and no timesheets
                continue
            task._fsm_ensure_sale_order()
            if task.sudo().sale_order_id.state in ['draft', 'sent']:
                task.sudo().sale_order_id.action_confirm()
                
            if task.stage_id == self.env.ref('access_protection.sav_effectue', False):                
                task.stage_id = self.env.ref('access_protection.sav_termine_f').id
            
            
    def action_timer_start(self):
        if not self.user_timer_id.timer_start and self.display_timesheet_timer:
            super(Task, self).action_timer_start()
            
        if not self.timer_start_ap:
            self.timer_start_ap = self.timer_start
        if not self.ap_date_inter_confirme_par_le_client:
            self.ap_date_inter_confirme_par_le_client = True
        if not self.user_id:
            self.user_id = self.env.user.id
            
        stage_planifie_sav = self.env.ref('access_protection.sav_planifie')
        if stage_planifie_sav:
            self.stage_id = stage_planifie_sav.id
            
    def action_timer_stop(self):
        if self.user_timer_id.timer_start and self.display_timesheet_timer:
            self.timer_stop_ap = fields.Datetime.now()
        return super(Task, self).action_timer_stop()
    
    @api.depends('stage_id')
    def _compute_display_terminer_buttons(self):
        for task in self:           
            if task.stage_id == self.env.ref('access_protection.sav_effectue', False) and self.env.user.has_group('access_protection.group_user_ap_manager'):
                task.display_termine_nf = True
            else:
                task.display_termine_nf = False
                
    def action_termine_nf(self):
        for task in self:
            if task.stage_id == self.env.ref('access_protection.sav_effectue', False):
                task.stage_id = self.env.ref('access_protection.sav_termine_nf').id
